/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Collections;
import java.util.PriorityQueue;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;
import org.apache.lucene.util.RandomAccessInputStream;

public class FileSorter {
    public static final int DEFAULT_MEM_LIMIT = 0xA00000;
    private File tmpFile;
    private int memLimit;
    private int nLinesAdded;
    private int curBlockMem = 0;
    private ArrayList curBlockLines = new ArrayList();
    private ArrayList blockOffsets = new ArrayList();
    private static String SENTINEL = "\ueeee\ueede\ueee1";

    protected FileSorter() {
    }

    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: sort <inFile> <outFile>");
            System.exit(1);
        }
        try {
            long startTime = System.currentTimeMillis();
            FileSorter.sort(new File(args[0]), new File(args[1]));
            System.out.println("Sort time: " + (float)(System.currentTimeMillis() - startTime) / 1000.0f + " sec");
        }
        catch (IOException e) {
            System.err.println(e);
        }
    }

    public static void sort(File inFile, File outFile) throws IOException {
        FileSorter.sort(inFile, outFile, null, 0xA00000);
    }

    public static void sort(File inFile, File outFile, File tmpDir, int memLimit) throws IOException {
        FileSorter.clearFile(outFile);
        BufferedReader in = new BufferedReader(new FileReader(inFile));
        try {
            try {
                String line;
                FileSorter sorter = FileSorter.start(tmpDir, memLimit);
                while ((line = in.readLine()) != null) {
                    sorter.addLine(line);
                }
                sorter.finish(new FileOutput(outFile, memLimit / 10));
            }
            catch (IOException e) {
                outFile.delete();
                throw e;
            }
        }
        finally {
            in.close();
        }
    }

    public static FileSorter start(File tmpDir, int memLimit) throws IOException {
        if (tmpDir != null && !tmpDir.isDirectory()) {
            throw new IOException("Invalid temp directory specified");
        }
        FileSorter sorter = new FileSorter();
        sorter.memLimit = memLimit;
        sorter.tmpFile = File.createTempFile("sort", ".tmp", tmpDir);
        return sorter;
    }

    public void addLine(String line) throws IOException {
        this.curBlockLines.add(line);
        ++this.nLinesAdded;
        this.curBlockMem += FileSorter.memSize(line);
        if (this.curBlockMem >= this.memLimit) {
            this.flushBlock();
        }
    }

    public int nLinesAdded() {
        return this.nLinesAdded;
    }

    public void finish(Output out) throws IOException {
        if (this.blockOffsets.isEmpty()) {
            Collections.sort(this.curBlockLines);
            int i = 0;
            while (i < this.curBlockLines.size()) {
                out.writeLine((String)this.curBlockLines.get(i));
                ++i;
            }
            out.close();
            FileSorter.clearFile(this.tmpFile);
            return;
        }
        this.flushBlock();
        int blockMemLimit = Math.max(16384, this.memLimit / this.blockOffsets.size());
        RandomAccessFile tmpIn = new RandomAccessFile(this.tmpFile, "r");
        try {
            PriorityQueue<BlockReader> queue = new PriorityQueue<BlockReader>(this.blockOffsets.size());
            int i = 0;
            while (i < this.blockOffsets.size()) {
                long blockPos = (Long)this.blockOffsets.get(i);
                BlockReader block = new BlockReader(tmpIn, blockPos, blockMemLimit);
                if (block.next()) {
                    queue.add(block);
                }
                ++i;
            }
            String prev = "";
            int nLinesWritten = 0;
            while (!queue.isEmpty()) {
                BlockReader block = (BlockReader)queue.remove();
                String line = block.cur();
                assert (line.compareTo(prev) >= 0) : "merge or sort algorithm failed";
                prev = line;
                out.writeLine(line);
                ++nLinesWritten;
                if (!block.next()) continue;
                queue.add(block);
            }
            assert (nLinesWritten == this.nLinesAdded) : "wrong number of lines written";
        }
        finally {
            out.close();
            tmpIn.close();
            FileSorter.clearFile(this.tmpFile);
        }
    }

    private void flushBlock() throws IOException {
        Collections.sort(this.curBlockLines);
        this.blockOffsets.add(new Long(this.tmpFile.length()));
        FileOutputStream tmpOut = new FileOutputStream(this.tmpFile, true);
        try {
            DeflaterOutputStream deflater = new DeflaterOutputStream(tmpOut);
            DataOutputStream blockOut = new DataOutputStream(deflater);
            int i = 0;
            while (i < this.curBlockLines.size()) {
                blockOut.writeUTF((String)this.curBlockLines.get(i));
                ++i;
            }
            blockOut.writeUTF(SENTINEL);
            blockOut.flush();
            deflater.finish();
            this.curBlockLines.clear();
            this.curBlockMem = 0;
        }
        finally {
            tmpOut.close();
        }
    }

    private static void clearFile(File f) throws IOException {
        if (!f.canRead()) {
            return;
        }
        if (f.delete()) {
            return;
        }
        FileOutputStream truncator = new FileOutputStream(f);
        truncator.close();
        f.delete();
    }

    private static int memSize(String s) {
        return s.length() * 2 + 32;
    }

    private static class BlockReader
    implements Comparable {
        RandomAccessFile base;
        DataInput in;
        long pos;
        long memLimit;
        ArrayList buffer = new ArrayList();
        int cur = -1;
        boolean eof = false;

        public BlockReader(RandomAccessFile base, long pos, int memLimit) throws IOException {
            this.base = base;
            this.pos = pos;
            this.memLimit = memLimit;
            this.in = new DataInputStream(new InflaterInputStream(new RandomAccessInputStream(base)));
        }

        public boolean next() throws IOException {
            ++this.cur;
            return this.cur != this.buffer.size() || this.fill();
        }

        public String cur() {
            return (String)this.buffer.get(this.cur);
        }

        public int compareTo(Object other) {
            return this.cur().compareTo(((BlockReader)other).cur());
        }

        private boolean fill() throws IOException {
            this.buffer.clear();
            if (this.eof) {
                return false;
            }
            this.base.seek(this.pos);
            long memUsed = 0L;
            while (memUsed < this.memLimit) {
                String line = this.in.readUTF();
                if (line.equals(SENTINEL)) {
                    this.eof = true;
                    break;
                }
                this.buffer.add(line);
                memUsed += (long)FileSorter.memSize(line);
            }
            this.pos = this.base.getFilePointer();
            this.cur = 0;
            return memUsed > 0L;
        }
    }

    public static class FileOutput
    implements Output {
        private BufferedWriter out;

        public FileOutput(File f, int bufSize) throws IOException {
            this.out = new BufferedWriter(new FileWriter(f), bufSize);
        }

        public void writeLine(String s) throws IOException {
            this.out.write(s);
            this.out.write(10);
        }

        public void close() throws IOException {
            this.out.close();
        }
    }

    public static interface Output {
        public void writeLine(String var1) throws IOException;

        public void close() throws IOException;
    }
}

